---@module "factorio_runtime"



---@class PlayerDB
---@field gui_ntt LuaEntity|LuaItem|LuaInventory?
---@field gui_type string?
---@field open_config boolean
---@field pin_top boolean
---@field pos_default [integer,integer]
---@field pos_data { typ:[integer,integer], type:[integer,integer], name:[integer,integer] }

---@alias foreground_gui_DB { [LuaGuiElement]: LuaGuiElement }	# table of gui frame to gui topper

---@alias storage_DB_format { foreground_gui:foreground_gui_DB, [integer]:PlayerDB }



--#region localised_globals
local mod_name = script.mod_name

local pairs = pairs
local tonumber = tonumber
local type = type
local xpcall = xpcall

local re_match = string.match
local str_len = string.len
local str_substr = string.sub

local arn_concat = table.concat
local arn_pack = table.pack
local arn_unpack = table.unpack
local arr_insert = table.insert
local arr_remove = table.remove
local arr_sort = table.sort
local tbl_size = table_size

local arr_empty = {}
--#endregion localised_globals



--#region debug_functions

---@param ... string
---@return string
local function safecall_err_handler(...)
	return arn_concat({
		"safecall error:",
		arn_concat({ ... }, "\n"),
		debug.traceback()
	}, "\n")
end

---@generic T0
---@generic T1
---@generic T2
---@generic T3
---@generic T4
---@generic T5
---@generic T6
---@generic T7
---@generic T8
---@generic T9
---@param func function<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9>
---@param ... any
---@return T0?
---@return T1?
---@return T2?
---@return T3?
---@return T4?
---@return T5?
---@return T6?
---@return T7?
---@return T8?
---@return T9?
local function safecall(func, ...)
	local ret = arn_pack(xpcall(func, safecall_err_handler, ...))
	if ret[1] then return arn_unpack(ret, 2, ret["n"]) end
	log(arn_concat(ret, "\n", 2, ret["n"]))
	game.print { mod_name .. ".safecall-error", { "mod-name." .. mod_name }, mod_name }
end

---@generic T0
---@generic T1
---@generic T2
---@generic T3
---@generic T4
---@generic T5
---@generic T6
---@generic T7
---@generic T8
---@generic T9
---@param func function<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9>
---@return function<T0,T1,T2,T3,T4,T5,T6,T7,T8,T9>
local function safecall_wrapper(func)
	return function(...)
		return safecall(func, ...)
	end
end
--#endregion debug_functions



--#region utility_functions
---returns default if value is not the same type as default
---@generic T
---@param value any
---@param default T
---@return T
local function default_value(value, default)
	if type(default) == type(value) then
		return value
	end
	return default
end

---split version string into number blocks
---@param txt string
---@return [integer,integer,integer] ver	easily comparable version
local function hlp_ver_explode(txt)
	local ver = {re_match(txt,"^(%d+)%.(%d+)%.(%d+)$")}
	for k,v in pairs(ver) do
		ver[k] = tonumber(v,10)
	end
	return ver
end

---compare equal length arrays
---@generic T
---@param a T[]
---@param b T[]
---@return boolean? diff	true if a is less, false if a is more, nil if equal
local function cmp_arr(a,b)
	for k,v in pairs(a) do
		local w = b[k]
		if v ~= w then return v < w end
	end
	return nil
end

---convert position to array
---@param pos ([number,number]|{x:number,y:number})?
---@return [number,number] pos	simplified position
local function hlp_pos_uniform(pos)
	if not pos then return {0,0} end
	return {pos.x or pos[1] or 0,pos.y or pos[2] or 0}
end

---inventory iter for for loop
---@param inv LuaInventory inventory to loop
---@param slot integer
---@return integer? slot
---@return LuaItem? item
local function inv_items_prev(inv,slot)
	while slot > 1 do
		slot = slot - 1
		---@type any
		local i = inv[slot]
		---@cast i LuaItemStack
		if i.valid_for_read then
			i = i.item
			---@cast i LuaItem?
			if i then
				return slot,i
			end
		end
	end
end

---inventory iter for for loop
---@param inv LuaInventory? inventory to loop
---@return fun(LuaInventory,integer):integer?,LuaItem? inv_items_prev
---@return LuaInventory? inv
---@return integer max
local function inv_items(inv)
	return inv_items_prev,inv,inv and (#inv +1) or 0
end
--#endregion utility_functions



--#region table_functions
---gets a subtable from a table, creating and assigning subtable if it does not exist
---@param tbl table table to get subtable from
---@param key any key in table
---@return table subtable the queried or new subtable
local function tbl_get_tbl(tbl, key)
	---@type table
	local ret = tbl[key]
	if type(ret) ~= "table" then
		ret = {}
		tbl[key] = ret
	end
	return ret
end

---returns false for tables and table like objects
---@param obj any
---@return boolean
local function is_not_table(obj) -- returns false for table like objects
	local t = type(obj)
	return t ~= "table" and (t ~= "userdata" or obj.object_name ~= "LuaCustomTable")
end

---compares values, follows recursively into tables to check if they have equal key value pairs
---@param ... any # two values to compare against each other
---@return boolean # true if they are equal
local function tbl_eq(...)
	---@type table<table|LuaCustomTable,table<table|LuaCustomTable,boolean>>
	local lookup = {}
	local function teq(a, b)
		if a == b then return true end                                                 -- short circuit on the same table/value
		if is_not_table(a) or is_not_table(b) then return false end                    -- only if both are tables they can be equal
		if lookup[a] and lookup[a][b] or lookup[b] and lookup[b][a] then return true end -- recursive tables already under investigation are equal until proven otherwise
		if lookup[a] then                                                              -- keep lookup small
			lookup[a][b] = true
		elseif lookup[b] then                                                          -- keep lookup small
			lookup[b][a] = true
		else
			lookup[a] = { [b] = true }
		end
		for k, v in pairs(a) do
			if not teq(b[k], v) then
				return false
			end
		end
		for k, v in pairs(b) do
			if a[k] == nil and v ~= nil then
				return false
			end
		end
		return true
	end
	return teq(...)
end

---adds a value to an array, if an [equal](lua://tbl_eq) value is not found.
---@generic T # type of array values
---@param arr T[] # array to add to
---@param val T # value to add
---@return T val # value found in array or the new value
local function arr_add_new(arr, val)
	for _, v in pairs(arr) do
		if tbl_eq(v, val) then return v end
	end
	arr_insert(arr, val)
	return val
end

---@generic K
---@generic V
---@param tbl table<K,V>
---@return table<V,K>
local function tbl_flip(tbl)
	local r = {}
	for k, v in pairs(tbl) do
		r[v] = k
	end
	return r
end
--#endregion table_functions



--#region self_accellerating_functions
---self accelerating game.get_player
---@param ... string|uint # id or name of the player
---@return LuaPlayer # the found player
local function get_player(...)
	get_player = game.get_player
	return get_player(...)
end

---get player mod setting
---@type fun(pid:integer,var:string):Color|boolean|number|integer|string|nil
local cfg_player_get
---set player mod setting
---@type fun(pid:integer,var:string,val:Color|boolean|number|integer|string):nil
local cfg_player_set
---get player mod setting circumventing cache
---@type fun(pid:integer,full_var:string):Color|boolean|number|integer|string|nil
local cfg_player_get_raw

local cfg_player_prefix = mod_name.."-"
local cfg_player_prefix_len = str_len(cfg_player_prefix)

do
---@type function
local cfg_player_getter
---@type {[integer]:{[string]:Color|boolean|number|integer|string}}
local cfg_player_cache = {}

---get player mod settings cache
---@param pid integer
---@return {[string]:Color|boolean|number|integer|string}
local function cfg_player_get_cache(pid)
	local ret = cfg_player_cache[pid]
	if ret then return ret end
	ret = {}
	for k,v in pairs(cfg_player_getter(pid)) do
		if str_substr(k,1,cfg_player_prefix_len) == cfg_player_prefix then
			ret[str_substr(k,cfg_player_prefix_len+1)] = v.value
		end
	end
	cfg_player_cache[pid] = ret
	return ret
end

---get player mod setting
---@param pid integer
---@param var string
---@return Color|boolean|number|integer|string|nil
local function cfg_player_get_real(pid,var)
	return cfg_player_get_cache(pid)[var]
end

---set player mod setting
---@param pid integer
---@param var string
---@param val Color|boolean|number|integer|string
---@return nil
local function cfg_player_set_real(pid,var,val)
	local c = cfg_player_get_cache(pid)
	if c[var] == val then return end
	c[var] = val
	cfg_player_getter(pid)[arn_concat{mod_name,"-",var}] = {value=val}
end

---get player mod setting circumventing cache
---@param pid integer
---@param full_var string
---@return Color|boolean|number|integer|string|nil
local function cfg_player_get_raw_real(pid,full_var)
	local val = cfg_player_getter(pid)[full_var].value
	if str_substr(full_var,1,cfg_player_prefix_len) == cfg_player_prefix then
		cfg_player_get_cache(pid)[str_substr(full_var,cfg_player_prefix_len+1)] = val
	end
	return val
end

---@type fun(pid:integer,var:string):Color|boolean|number|integer|string|nil
function cfg_player_get(...)
	cfg_player_getter = settings.get_player_settings
	cfg_player_get = cfg_player_get_real
	cfg_player_set = cfg_player_set_real
	cfg_player_get_raw = cfg_player_get_raw_real
	return cfg_player_get(...)
end
---@type fun(pid:integer,var:string,val:Color|boolean|number|integer|string):nil
function cfg_player_set(...)
	cfg_player_getter = settings.get_player_settings
	cfg_player_get = cfg_player_get_real
	cfg_player_set = cfg_player_set_real
	cfg_player_get_raw = cfg_player_get_raw_real
	return cfg_player_set(...)
end
---@type fun(pid:integer,full_var:string):Color|boolean|number|integer|string|nil
function cfg_player_get_raw(...)
	cfg_player_getter = settings.get_player_settings
	cfg_player_get = cfg_player_get_real
	cfg_player_set = cfg_player_set_real
	cfg_player_get_raw = cfg_player_get_raw_real
	return cfg_player_get_raw(...)
end
end

---get a db table from storage
---@return nil
---@overload fun(key:any,delete:boolean):nil	# delete instead of return
---@overload fun(key:any):table	# get db from storage
local function get_db(...)
	---@type storage_DB_format
	local db = storage
	if type(db) ~= "table" then
		db = {}
		storage = db
	end
	---get a db table from storage
	---@param key any	# key for the data to retrieve
	---@param delete boolean	# delete instead of return
	---@return nil
	---@overload fun(key:any):table	# get db from storage
	function get_db(key,delete)
		if delete ~= true then
			return tbl_get_tbl(db,key)
		end
		db[key] = nil
	end
	return get_db(...)
end
--#endregion self_accellerating_functions









-- circumventing factorio quirks --

local cfg_defines_inventory_item_main = defines.inventory.item_main
local cfg_record_name_to_type = {
	["blueprint"] = "blueprint",
	["blueprint-book"] = "blueprint-book",
	["deconstruction-planner"] = "deconstruction-item",
	["upgrade-planner"] = "upgrade-item",
}
local get_prototype_type
do
	---@type { [string]: (fun(obj:LuaItemStack|LuaRecord|LuaInventory|LuaEntity|LuaGuiElement|LuaItem|LuaSurface):string,string) }
	local get_prototype_type_func = {
		---@param obj LuaItemStack
		["LuaItemStack"] = function(obj) return obj.type, obj.name end,
		---@param obj LuaRecord
		["LuaRecord"] = function(obj) return cfg_record_name_to_type[obj.type], obj.type end,
		["LuaInventory"] = function() return "LuaInventory", "LuaInventory" end,
	}
	for _, v in pairs { "LuaEntity", "LuaGuiElement", "LuaItem", "LuaSurface" } do
		get_prototype_type_func[v] = get_prototype_type_func["LuaItemStack"]
	end
	---@param obj LuaItemStack|LuaRecord|LuaInventory|LuaEntity|LuaGuiElement|LuaItem|LuaSurface
	---@return string type	# prototype type
	---@return string name	# prototype name
	function get_prototype_type(obj)
		return get_prototype_type_func[obj.object_name](obj)
	end
end

---get player db from storage
---@param pid integer	# player index
---@return PlayerDB
local function get_db_player(pid)
	---@type PlayerDB
	local db = get_db(pid)
	if db.gui_ntt and not db.gui_ntt.valid then
		db.gui_ntt = nil
		db.gui_type = nil
	end
	if type(db.open_config) ~= "boolean" then
		db.open_config = false
	end
	if type(db.pin_top) ~= "boolean" then
		db.pin_top = false
	end
	if type(db.pos_default) ~= "table" then
		db.pos_default = {0,0}
	end
	if type(db.pos_data) ~= "table" then
		db.pos_data = {["typ"]={},["type"]={},["name"]={}}
	end
	return db
end

---set gui data for player
---@param pid integer	# player index
---@param ntt LuaEntity|LuaItem|LuaInventory|nil
---@param typ string?
---@return nil
local function gui_db_player_set(pid,ntt,typ)
	local db = get_db_player(pid)
	db.gui_ntt = ntt
	db.gui_type = typ
end

---get gui data for player
---@param pid integer	# player index
---@return LuaEntity|LuaItem|LuaInventory|nil ntt
---@return string|nil typ
local function gui_db_player_get(pid)
	local db = get_db_player(pid)
	return db.gui_ntt,db.gui_type
end

---@param inv LuaInventory
---@return LuaItemStack[]
local function get_inventory_items(inv)
	---@type LuaItemStack[]
	local ret = {}
	for n = 1, #inv, 1 do
		---@type LuaItemStack
		local itm = inv[n]
		if itm and itm.valid_for_read then
			arr_insert(ret, itm)
		end
	end
	return ret
end

---@param ntt LuaEntity
---@param ... integer
---@return LuaInventory?
local function get_first_valid_inventory(ntt, ...)
	local get = ntt.get_inventory
	for _, n in pairs { ... } do
		local i = get(n)
		if i then return i end
	end
end

local function get_logistic_sections(ntt)
	local logi = ntt.get_logistic_sections()
	return logi and logi.sections or arr_empty
end

local has_sortable_logistic_sections
do
	local cfg_defines_logistic_section_type_transitional_request_controlled = defines.logistic_section_type.transitional_request_controlled
	has_sortable_logistic_sections = function(ntt)
		local logi = ntt.get_logistic_sections()
		if not logi then return false end
		logi = logi.sections
		if not logi then return false end
		for _, sect in pairs(logi) do
			if sect.type == cfg_defines_logistic_section_type_transitional_request_controlled then return false end
		end
		return true
	end
end

-- end of utilities --









-- compare functions --

local function cmp_str(a, b)
	local x = a or ""
	local y = b or ""
	if x > y then return false end
	if x < y then return true end
	return nil
end

local cmp_book_entries
do
	local cmp_book_entries_order = tbl_flip {
		"blueprint-book",
		"blueprint",
		"deconstruction-item",
		"upgrade-item",
	}
	---@param a LuaItemStack
	---@param b LuaItemStack
	---@return boolean?
	function cmp_book_entries(a, b)
		local x = cmp_book_entries_order[a.type] or 99
		local y = cmp_book_entries_order[b.type] or 99
		if x ~= y then return x < y end
	end
end

local cmp_comparator
do
	local cmp_comparator_order = {}
	do
		for k, v in pairs { { "<" }, { "<=", "≤" }, { ">=", "≥" }, { ">" }, { "!=", "≠" }, { "=" } } do
			for _, c in pairs(v) do
				cmp_comparator_order[c] = k
			end
		end
	end
	function cmp_comparator(a, b)
		local x = cmp_comparator_order[a] or 0
		local y = cmp_comparator_order[b] or 0
		if x ~= y then return x < y end
	end
end

local function cmp_prototype_lookup(...)
	local lookup = {}
	do -- nothing in here shall generate upvalues
		local prototypes = prototypes
		local order = {}
		for proto_type, proto_key in pairs {
			["asteroid-chunk"] = "asteroid_chunk",
			["entity"] = "entity",
			["equipment"] = "equipment",
			["fluid"] = "fluid",
			["item"] = "item",
			["quality"] = "quality",
			["recipe"] = "recipe",
			["space-connection"] = "space_connection",
			["space-location"] = "space_location",
			["technology"] = "technology",
			["tile"] = "tile",
			["virtual"] = "virtual_signal",
		} do
			lookup[proto_type] = {}
			for proto_name, prototype in pairs(prototypes[proto_key]) do
				local itm = prototypes["item"][proto_name]
				arr_insert(order, {
					prototype.group.order or "",
					prototype.group.name or "",
					prototype.subgroup.order or "",
					prototype.subgroup.name or "",
					itm and itm.order or prototype.order or "",
					proto_name,
					proto_type,
				})
			end
		end
		arr_sort(order, cmp_arr)
		for k, v in pairs(order) do
			lookup[v[7]][v[6]] = k
		end
	end -- nothing in here shall generate upvalues
	function cmp_prototype_lookup(name)
		return name and lookup[name] or lookup
	end

	return cmp_prototype_lookup(...)
end
local function cmp_prototype(...)
	local lookup = cmp_prototype_lookup()
	function cmp_prototype(aType, aName, bType, bName)
		if aName == bName and aType == bType then return nil end                 -- short circuit if equal
		if not (aType and aName) then return bType and bName and true or false end -- short circuit if a is nil
		if not (bType and bName) then return false end                           -- short circuit if b is nil
		local x, y = lookup[aType][aName], lookup[bType][bName]
		if x then
			return y and x < y or false
		end
		return y and true or nil
	end

	return cmp_prototype(...)
end
local function cmp_quality(...)
	local lookup = cmp_prototype_lookup("quality")
	function cmp_quality(aName, bName)
		if aName == bName then return nil end              -- short circuit if equal
		if not aName then return bName and true or false end -- short circuit if a is nil
		if not bName then return false end                 -- short circuit if b is nil
		local x, y = lookup[aName], lookup[bName]
		if x then
			return y and x < y or false
		end
		return y and true or nil
	end

	return cmp_quality(...)
end

-- sort functions --

---@param a LuaItemStack
---@param b LuaItemStack
---@return boolean?
local function sort_book_entries(a, b)
	local c = cmp_book_entries(a, b)
	if c ~= nil then return c end
	c = cmp_prototype("item", a.name, "item", b.name)
	if c ~= nil then return c end
	c = cmp_quality(a.quality.name, b.quality.name)
	if c ~= nil then return c end
	c = cmp_str(a.label, b.label)
	if c ~= nil then return c end
	return a.item_number < b.item_number
end
local function sort_logistics(a, b)
	local x = a.value
	local y = b.value
	local c = cmp_prototype(x.type, x.name, y.type, y.name)
	if c ~= nil then return c end
	c = cmp_comparator(x.comparator, y.comparator)
	if c ~= nil then return c end
	return cmp_quality(x.quality, y.quality)
end
local function sort_inf_filter(a, b)
	local c = cmp_prototype("item", a.name, "item", b.name)
	if c ~= nil then return c end
	c = cmp_str(a.mode, b.mode)
	if c ~= nil then return c end
	return cmp_quality(a.quality, b.quality)
end
local function sort_decon_filter(a, b)
	local c = cmp_prototype("entity", a.name, "entity", b.name)
	if c ~= nil then return c end
	c = cmp_comparator(a.comparator, b.comparator)
	if c ~= nil then return c end
	return cmp_quality(a.quality, b.quality)
end
local function sort_decon_tile_filter(a, b)
	return cmp_prototype("tile", a, "tile", b)
end
local function sort_upgrade_filter(a, b)
	local c = cmp_prototype(a.type, a.name, b.type, b.name)
	if c ~= nil then return c end
	c = cmp_comparator(a.comparator, b.comparator)
	if c ~= nil then return c end
	return cmp_quality(a.quality, b.quality)
end



-- configuration helper --

local cfg_has_infinity_filter = {
	["infinity-cargo-wagon"] = true,
	["infinity-container"] = true,
}

local cfg_inventories = {
	["assembling-machine"] = { defines.inventory.fuel, defines.inventory.assembling_machine_modules },
	["asteroid-collector"] = { defines.inventory.chest, nil },
	["beacon"] = { nil, defines.inventory.beacon_modules },
	["boiler"] = { defines.inventory.fuel, defines.inventory.burnt_result },
	["burner-generator"] = { defines.inventory.fuel, defines.inventory.burnt_result },
	["car"] = { defines.inventory.fuel, defines.inventory.car_trunk, defines.inventory.car_trash },
	["cargo-landing-pad"] = { defines.inventory.cargo_landing_pad_main, defines.inventory.cargo_landing_pad_trash },
	["cargo-pod"] = { defines.inventory.cargo_unit, nil },
	["cargo-wagon"] = { defines.inventory.cargo_wagon },
	["character"] = { defines.inventory.character_main, defines.inventory.character_trash, defines.inventory.character_vehicle },
	["character-corpse"] = { defines.inventory.character_corpse, nil },
	["container"] = { defines.inventory.chest, nil },
	["furnace"] = { defines.inventory.fuel, defines.inventory.furnace_modules },
	["fusion-reactor"] = { defines.inventory.fuel, defines.inventory.burnt_result },
	["infinity-cargo-wagon"] = { defines.inventory.cargo_wagon },
	["infinity-container"] = { defines.inventory.chest, defines.inventory.logistic_container_trash },
	["lab"] = { defines.inventory.fuel, defines.inventory.lab_modules },
	["linked-container"] = { defines.inventory.chest, nil },
	["locomotive"] = { defines.inventory.fuel, nil, nil },
	["logistic-container"] = { defines.inventory.chest, defines.inventory.logistic_container_trash },
	["market"] = { defines.inventory.chest, nil },
	["mining-drill"] = { defines.inventory.fuel, defines.inventory.mining_drill_modules },
	["proxy-container"] = { defines.inventory.proxy_main, nil },
	["reactor"] = { defines.inventory.fuel, defines.inventory.burnt_result },
	["roboport"] = { defines.inventory.roboport_robot, defines.inventory.roboport_material },
	["rocket-silo"] = { defines.inventory.fuel, defines.inventory.rocket_silo_rocket, defines.inventory.rocket_silo_trash, defines.inventory.rocket_silo_modules },
	["space-platform-hub"] = { defines.inventory.hub_main, defines.inventory.hub_trash },
	["spider-vehicle"] = { defines.inventory.fuel, defines.inventory.spider_trunk, defines.inventory.spider_trash },
	["temporary-container"] = { defines.inventory.chest, nil },
}
local cfg_type_map = {
	["assembling-machine"] = "crafter",
	["asteroid-collector"] = "chest",
	["beacon"] = "crafter",
	["blueprint-book"] = "book",
	["boiler"] = "reactor",
	["burner-generator"] = "reactor",
	["car"] = "car",
	["cargo-landing-pad"] = "space",
	["cargo-pod"] = "chest",
	["cargo-wagon"] = "train",
	["character"] = "player",
	["character-corpse"] = "chest",
	["constant-combinator"] = "math",
	["container"] = "chest",
	["deconstruction-item"] = "decon",
	["furnace"] = "crafter",
	["fusion-reactor"] = "reactor",
	["infinity-cargo-wagon"] = "train",
	["infinity-container"] = "chest",
	["item-with-inventory"] = "item",
	["lab"] = "crafter",
	["linked-container"] = "chest",
	["locomotive"] = "car",
	["logistic-container"] = "chest",
	["market"] = "chest",
	["mining-drill"] = "crafter",
	["proxy-container"] = "chest",
	["reactor"] = "reactor",
	["roboport"] = "robo",
	["rocket-silo"] = "rocket",
	["space-platform-hub"] = "space",
	["spider-vehicle"] = "car",
	["temporary-container"] = "chest",
	["upgrade-item"] = "upgrade",
}
local cfg_inv_options = {
	"car1",    -- fuel
	"car2",    -- main
	"car3",    -- trash
	"chest1",  -- main
	"chest2",  -- trash
	"crafter1", -- fuel
	"crafter2", -- module
	"player1", -- main
	"player2", -- trash
	"player3", -- aux
	"reactor1", -- fuel
	"reactor2", -- waste
	"robo1",   -- main
	"robo2",   -- aux
	"rocket1", -- fuel
	"rocket2", -- main
	"rocket3", -- trash
	"rocket4", -- module
	"self1",   -- main
	"self2",   -- trash
	"self3",   -- aux
	"space1",  -- main
	"space2",  -- trash
	"train1",  -- main
}
local cfg_logi_options = {
	"car",
	"chest",
	"crafter",
	"math",
	"other",
	"player",
	"reactor",
	"robo",
	"rocket",
	"self",
	"space",
	"train",
}







-- sort inventories --

---@param ntt LuaEntity
---@param num integer
local function handle_entity_inventory_num(ntt, num)
	---@type any
	local inv = cfg_inventories[ntt.type]
	if not inv then return end
	inv = inv[num]
	if not inv then return end
	inv = ntt.get_inventory(inv)
	if not inv then return end
	inv.sort_and_merge()
end

---@param ntt LuaEntity
local function handle_entity_inventory_all(ntt)
	local get = ntt.get_inventory
	for _, n in pairs(cfg_inventories[ntt.type] or arr_empty) do
		local i = get(n)
		if i then
			i.sort_and_merge()
		end
	end
end

---@param ntt LuaEntity
---@param num integer
local function handle_entity_inventory(ntt, num)
	if num == nil then return handle_entity_inventory_all(ntt) end
	return handle_entity_inventory_num(ntt, num)
end

-- sort logistic section filters --

local function handle_entity_logistic_sections(ntt)
	for _, sect in pairs(get_logistic_sections(ntt)) do
		if sect.is_manual and sect.filters then
			local lst = {}
			for _, fltr in pairs(sect.filters) do
				if fltr.value then
					arr_add_new(lst, fltr)
				end
			end
			arr_sort(lst, sort_logistics)
			sect.filters = lst
		end
	end
end

-- sort infinity container filters --

local function handle_entity_infinity_filters(ntt)
	if cfg_has_infinity_filter[ntt.type] and ntt.infinity_container_filters then
		local lst = {}
		for _, fltr in pairs(ntt.infinity_container_filters) do
			if fltr.name then
				fltr.index = nil
				arr_add_new(lst, fltr)
			end
		end
		arr_sort(lst, sort_inf_filter)
		for i, fltr in pairs(lst) do
			fltr.index = i
		end
		ntt.infinity_container_filters = lst
	end
end

-- sort deconstruction planner --

---sorts deconstruction planner
---@param itm LuaItem|LuaItemStack|LuaRecord
local function do_sort_decon_planner(itm)
	local lst = {}
	for _, fltr in pairs(itm.entity_filters) do
		arr_add_new(lst, fltr)
	end
	arr_sort(lst, sort_decon_filter)
	itm.entity_filters = lst
	-- also sort tiles
	lst = {}
	for _, fltr in pairs(itm.tile_filters) do
		arr_add_new(lst, fltr)
	end
	arr_sort(lst, sort_decon_tile_filter)
	itm.tile_filters = lst
end

-- sort upgrade planner --

---sorts upgrade planner
---@param itm LuaItem|LuaItemStack|LuaRecord
local function do_sort_upgrade_planner(itm)
	local from = {}
	local to = {}
	local to_only = {}
	local fn = itm.get_mapper
	for i = 1, itm.mapper_count do
		---@type UpgradeMapperSource|nil
		local s = fn(i, "from") --[[@as UpgradeMapperSource]]
		---@type UpgradeMapperDestination|nil
		local d = fn(i, "to") --[[@as UpgradeMapperDestination]]
		if s and not s.name then s = nil end
		if d and not d.name then d = nil end
		if s then
			to[arr_add_new(from, s)] = d
		elseif d then
			arr_add_new(to_only, d)
		end
	end
	arr_sort(from, sort_upgrade_filter)
	arr_sort(to_only, sort_upgrade_filter)
	fn = itm.set_mapper
	for i = 1, #from do
		fn(i, "from", from[i])
		fn(i, "to", to[from[i]])
	end
	for i = 1, #to_only do
		fn(i + #from, "from", nil)
		fn(i + #from, "to", to_only[i])
	end
	for i = 1 + #from + #to_only, itm.mapper_count do
		fn(i, "from", nil)
		fn(i, "to", nil)
	end
end

-- sort blueprint book --

---sort entries of book
---@param itm LuaItem|LuaItemStack|LuaRecord
local function do_sort_book(itm)
	if itm and itm.object_name == "LuaItemStack" then itm = itm.item end
	if not itm or itm.object_name ~= "LuaItem" then return end
	local inv = itm.get_inventory(cfg_defines_inventory_item_main)
	local lst = get_inventory_items(inv)
	arr_sort(lst, sort_book_entries)
	for k, v in pairs(lst) do
		---@cast lst LuaItem[]
		lst[k] = v.item
	end
	for i = #lst, 1, -1 do
		local src = lst[i]
		local dst = inv[i]
		if not dst.valid_for_read or dst.item_number ~= src.item_number then
			if not dst.swap_stack(src.item_stack) then
				return
			end
		end
	end
end

---recursively sort children of book
---@param pid integer player id
---@param itm LuaItem|LuaItemStack|LuaRecord
---@param full boolean? ignore config and force sort
local function do_sort_books(pid,itm,full)
	local cond = {
		["book"] = do_sort_book,
		["decon"] = do_sort_decon_planner,
		["upgrade"] = do_sort_upgrade_planner,
	}
	for k in pairs(cond) do
		if not full and not default_value(cfg_player_get(pid,"sort-"..k),false) then
			cond[k] = nil
		end
	end
	if tbl_size(cond) < 1 then return end
	---@param book LuaRecord
	local function recurseR(book)
		for _,rec in pairs(book.contents) do
			local t = cfg_type_map[cfg_record_name_to_type[rec.type]]
			local f = cond[t]
			if f then f(rec) end
			if t == "book" then
				recurseR(rec)
			end
		end
	end
	---@param book LuaItem|LuaItemStack
	local function recurseI(book)
		for _,rec in inv_items(book.get_inventory(cfg_defines_inventory_item_main)) do
			local t = cfg_type_map[rec.type]
			local f = cond[t]
			if f then f(rec) end
			if t == "book" then
				recurseI(rec)
			end
		end
	end
	local tbl = {
		["LuaItem"] = recurseI,
		["LuaItemStack"] = recurseI,
		["LuaRecord"] = recurseR,
	}
	return tbl[itm.object_name](itm)
end



local actn_sort_object
do
local fn_table_item = {
---@param pid integer player id
---@param obj LuaItem|LuaItemStack
---@param full boolean? only autosort or full
---@return nil
	["item-with-inventory"] = function(pid,obj,full)
		local typ = cfg_type_map[obj.type]
		if not typ or not full and not cfg_player_get(pid,"sort-"..typ) then return end
		obj.get_inventory(cfg_defines_inventory_item_main).sort_and_merge()
	end,
---@param pid integer player id
---@param obj LuaItem|LuaItemStack
---@param full boolean? only autosort or full
---@return nil
	["blueprint-book"] = function(pid,obj,full)
		local deep = cfg_player_get(pid,"sort-books")
		local typ = cfg_type_map[obj.type]
		if not typ or not full and not deep and not cfg_player_get(pid,"sort-"..typ) then return end
		if deep or full then
			do_sort_books(pid,obj,full)
		end
		return do_sort_book(obj)
	end,
---@param pid integer player id
---@param obj LuaItem|LuaItemStack
---@param full boolean? only autosort or full
---@return nil
	["deconstruction-item"] = function(pid,obj,full)
		local typ = cfg_type_map[obj.type]
		if not typ or not full and not cfg_player_get(pid,"sort-"..typ) then return end
		return do_sort_decon_planner(obj)
	end,
---@param pid integer player id
---@param obj LuaItem|LuaItemStack
---@param full boolean? only autosort or full
---@return nil
	["upgrade-item"] = function(pid,obj,full)
		local typ = cfg_type_map[obj.type]
		if not typ or not full and not cfg_player_get(pid,"sort-"..typ) then return end
		return do_sort_upgrade_planner(obj)
	end,
}
local fn_table_record = {
---@param pid integer player id
---@param obj LuaRecord
---@param full boolean? only autosort or full
---@return nil
	["blueprint-book"] = function(pid,obj,full)
		local deep = cfg_player_get(pid,"sort-books")
		local typ = cfg_type_map[cfg_record_name_to_type[obj.type]]
		if not typ or not full and not deep and not cfg_player_get(pid,"sort-"..typ) then return end
		if deep or full then
			do_sort_books(pid,obj,full)
		end
		return do_sort_book(obj)
	end,
---@param pid integer player id
---@param obj LuaRecord
---@param full boolean? only autosort or full
---@return nil
	["deconstruction-planner"] = function(pid,obj,full)
		local typ = cfg_type_map[cfg_record_name_to_type[obj.type]]
		if not typ or not full and not cfg_player_get(pid,"sort-"..typ) then return end
		return do_sort_decon_planner(obj)
	end,
---@param pid integer player id
---@param obj LuaRecord
---@param full boolean? only autosort or full
---@return nil
	["upgrade-planner"] = function(pid,obj,full)
		local typ = cfg_type_map[cfg_record_name_to_type[obj.type]]
		if not typ or not full and not cfg_player_get(pid,"sort-"..typ) then return end
		return do_sort_upgrade_planner(obj)
	end,
}
local fn_table = {
---@param pid integer player id
---@param obj LuaEntity
---@param full boolean? only autosort or full
---@return nil
	["LuaEntity"] = function(pid,obj,full)
		local ptyp = obj.type
		local typ = cfg_type_map[ptyp]
		if ptyp == "character" then
			typ = (get_player(pid).character.unit_number == obj.unit_number) and "self" or "player"
		end
		if typ then
			for k in pairs(cfg_inventories[ptyp] or arr_empty) do
				if full or cfg_player_get(pid,arn_concat{"sort-",typ,k}) then
					handle_entity_inventory(obj,k)
				end
			end
		end
		if full or cfg_player_get(pid,"sort-"..(typ or "other")) then
			handle_entity_logistic_sections(obj)
		end
		if cfg_has_infinity_filter[ptyp] and ( full or cfg_player_get(pid,"sort-infinity") ) then
			handle_entity_infinity_filters(obj)
		end
	end,
---@param pid integer player id
---@param obj LuaItem|LuaItemStack
---@param full boolean? only autosort or full
---@return nil
	["LuaItemStack"] = function(pid,obj,full)
		if not obj.valid_for_read then return end
		local fn = fn_table_item[obj.type]
		if fn then return fn(pid,obj,full) end
	end,
---@param pid integer player id
---@param obj LuaRecord
---@param full boolean? only autosort or full
---@return nil
	["LuaRecord"] = function(pid,obj,full)
		if not obj.valid_for_write then return end
		local fn = fn_table_record[obj.type]
		if fn then return fn(pid,obj,full) end
	end,
---@param pid integer player id
---@param obj LuaInventory
---@param full boolean? only autosort or full
---@return nil
	["LuaInventory"] = function(pid,obj,full)
		if not full and not cfg_player_get(pid,"sort-scriptinv") then return end
		obj.sort_and_merge()
	end,
}
fn_table["LuaItem"] = fn_table["LuaItemStack"]
---sort object
---@param pid integer player id
---@param obj LuaEntity|LuaInventory|LuaItem|LuaItemStack|LuaRecord
---@param full boolean? only autosort or full
---@return nil
function actn_sort_object(pid,obj,full)
	if not obj or not obj.valid then return end
	local fn = fn_table[obj.object_name]
	if fn then return fn(pid,obj,full) end
end
end







-- common events --

---sort target
---@param pid integer player id
---@param full boolean full sort or only autosort
---@return nil
local function evt_sort_target(pid,full)
	local plr = get_player(pid)
	local target = plr.selected
	if not target then return end
	return actn_sort_object(pid,target,full)
end

---sort hand
---@param pid integer player id
---@param full boolean full sort or only autosort
---@return nil
local function evt_sort_hand(pid,full)
	local plr = get_player(pid)
	---@type LuaItemStack|LuaRecord|nil
	local target = plr.cursor_stack
	if not target or not target.valid_for_read then
		target = plr.cursor_record
	end
	if not target then return end
	return actn_sort_object(pid,target,full)
end

---sort opened
---@param pid integer player id
---@param full boolean full sort or only autosort
---@return nil
local function evt_sort_opened(pid,full)
	local target = get_player(pid).opened
	if not target then return end
	return actn_sort_object(pid,target,full)
end

---sort self
---@param pid integer player id
---@param full boolean full sort or only autosort
---@return nil
local function evt_sort_self(pid,full)
	local plr = get_player(pid)
	actn_sort_object(pid,plr.vehicle,full)
	return actn_sort_object(pid,plr.character,full)
end

---syncs autosort elements to setting
---@param pid integer player id
---@param enable boolean set state
---@return nil
local function evt_sync_autosort(pid,enable)
	get_player(pid).set_shortcut_toggled("sbSortedLogistics-toggle",enable)
end

---set autosort state for player
---@param pid integer player id
---@param enable boolean? set state or toggle
---@return nil
local function evt_toggle_autosort(pid,enable)
	if enable == nil then
		enable = not default_value(cfg_player_get(pid,"enabled"),true)
	end
	cfg_player_set(pid,"enabled",enable)
end







-- gui functions --

local function start_foreground_gui()
	local db = get_db("foreground_gui")
	local function on_tick()
		local disable = true
		for window,pin in pairs(db) do
			if window.valid and window.visible and pin.valid and pin.toggled then
				window.bring_to_front()
				disable = false
			else
				db[window] = nil
			end
		end
		if disable then
			script.on_event(defines.events.on_tick,nil)
		end
	end
	function start_foreground_gui()
		script.on_event(defines.events.on_tick,on_tick)
	end
	start_foreground_gui()
end

---@param el LuaGuiElement
---@param name string
---@return LuaGuiElement?
local function gui_locate_child(el, name)
	if el.name == name then return el end
	for _, c in pairs(el.children) do
		local r = gui_locate_child(c, name)
		if r then return r end
	end
end

---@param el LuaGuiElement
---@param name string
---@return LuaGuiElement?
local function gui_locate_parent(el, name)
	while el and el.object_name == "LuaGuiElement" do
		if el.name == name then return el end
		el = el.parent
	end
end

---@param el LuaGuiElement
---@param func fun(el:LuaGuiElement):nil
local function gui_recurse_children(el, func)
	for _, v in pairs(el.children) do
		gui_recurse_children(v, func)
		func(v)
	end
end

local gui_enable_drag
do
	local gui_enable_drag_allowed_types = {
		["empty-widget"] = true,
		["flow"] = true,
		["frame"] = true,
		["label"] = true,
		["table"] = true,
	}
	---@param gui LuaGuiElement
	function gui_enable_drag(gui)
		---@param el LuaGuiElement
		local function rec(el)
			for _, v in pairs(el.children) do
				if gui_enable_drag_allowed_types[v.type] then
					v.drag_target = gui
				end
				rec(v)
			end
		end
		return rec(gui)
	end
end

---@param arr table[]
---@param direction GuiDirection
---@param size? integer
---@return nil
local function gui_add_spacer(arr, direction, size)
	if #arr < 1 or arr[#arr].type == "empty-widget" then return end
	local style = {}
	if direction == "horizontal" then
		style.height = 0
		if size == nil then
			style.horizontally_stretchable = true
		else
			style.width = size
		end
	else
		style.width = 0
		if size == nil then
			style.vertically_stretchable = true
		else
			style.height = size
		end
	end
	return arr_insert(arr, {
		type = "empty-widget",
		style_overwrite = style,
	})
end

---@param arr table
---@param name string
---@param actn table
---@return nil
local function gui_add_button(arr, name, actn)
	return arr_insert(arr, {
		type = "button",
		-- 		name="sbSortedLogistics_sort"..id,
		caption = { "sbSortedLogistics-name.btn-sort-" .. name },
		tooltip = { "sbSortedLogistics-desc.btn-sort-" .. name },
		style_overwrite = {
			horizontally_stretchable = true,
			horizontally_squashable = false,
		},
		tags = { ["on_click"] = actn },
	})
end

---@param arr table
---@param name string
---@param actn table
---@return nil
local function gui_add_min_button(arr, name, actn)
	return arr_insert(arr, {
		type = "button",
		caption = { "sbSortedLogistics-name.btn-sort-" .. name },
		tooltip = { "sbSortedLogistics-desc.btn-sort-" .. name },
		style_overwrite = {
			-- 			horizontally_stretchable = true,
			-- 			horizontally_squashable = false,
			-- 			padding = 0,
			minimal_width = 0,
			minimal_height = 0,
		},
		tags = { ["on_click"] = actn },
	})
end

local function gui_add_setting(plr, arr, name)
	return arr_insert(arr, {
		type = "checkbox",
		state = default_value(cfg_player_get(plr.index,"sort-"..name),false),
		caption = { "mod-setting-name.sbSortedLogistics-sort-" .. name },
		tooltip = { "mod-setting-description.sbSortedLogistics-sort-" .. name },
		tags = { ["on_click"] = { "set_config", "plr", "el", "sort-" .. name } },
	})
end

---@param arr table
---@param direction GuiDirection
---@return LuaGuiElement
local function gui_add_flow(arr, direction)
	local t = {
		type = "flow",
		direction = direction,
	}
	arr_insert(arr, t)
	return t
end

local function gui_destroy_deep(el)
	for _, e in pairs(el.children) do
		local g = e["sbSortedLogistics_gui"]
		if g then
			g.destroy()
		end
		gui_destroy_deep(e)
	end
end

---@param pid integer	# player index
---@param data { [string]: [integer,integer] }
---@param default [integer,integer]
---@param ntt LuaEntity|LuaItem|LuaInventory
---@param typ string
---@return [integer,integer]
local function gui_find_position(pid,data,default,ntt,typ)
	local which = cfg_player_get(pid,"gui-position")
	if which == "none" then return default end
	if which == "fix" then return default end
	if which == "typ" then return data["typ"] and data["typ"][typ] or default end
	local ptype,pname = get_prototype_type(ntt)
	if which == "type" then return data["type"] and data["type"][ptype] or default end
	if which == "name" then return data["name"] and data["name"][pname] or default end
	return default
end

---@param plr LuaPlayer
---@param ntt LuaEntity|LuaItem|LuaInventory
---@param typ string
---@return LuaGuiElement
local function gui_get_window(plr,ntt,typ)
	local pid = plr.index
	---@type LuaGuiElement
	local gui = plr.gui.screen["sbSortedLogistics_gui"]
	if gui then return gui end
	local db = get_db_player(pid)

	gui_destroy_deep(plr.gui)
	gui = plr.gui.screen.add{
		type = "frame",
		name = "sbSortedLogistics_gui",
		direction = "vertical",
		--caption={"mod-name.sbSortedLogistics"},
		auto_center = false,
	}
	gui.location = gui_find_position(pid,db.pos_data,db.pos_default,ntt,typ)

	---@type LuaGuiElement,LuaGuiElement,LuaGuiElement,LuaGuiElement
	local flow,strip,tbl,e
	strip = gui.add { type = "flow", direction = "horizontal" }
	e = strip.add { type = "label", style = "frame_title", caption = { "mod-name.sbSortedLogistics" } }
	e.style.top_margin = -3
	e = strip.add { type = "empty-widget", style = "draggable_space_header" }
	e.style.horizontally_stretchable = true
	--e.style.vertically_stretchable = true
	e.style.padding = 0
	e.style.margin = 0
	e.style.height = 24
	e.style.natural_height = 24
	e = strip.add { type = "sprite-button", style = "frame_action_button", sprite = "utility/empty_robot_material_slot", tooltip = { "sbSortedLogistics-desc.btn-config" } }
	e.tags = { ["on_click"] = { "btn_config", "el" } }
	e.auto_toggle = true
	e.toggled = db.open_config
	e = strip.add { type = "sprite-button", name = "btn_pin_top", style = "frame_action_button", sprite = "utility/track_button_white", tooltip = { "sbSortedLogistics-desc.btn-pin-top" } }
	e.tags = { ["on_click"] = { "btn_pin_top", "el" } }
	e.auto_toggle = true
	e.toggled = db.pin_top

	strip = gui.add { type = "flow", name = "config", direction = "vertical" }
	strip.style.horizontally_stretchable = true
	strip.style.vertically_stretchable = true
	flow = strip.add { type = "flow", direction = "horizontal" }
	for option, values in pairs { ["gui-show"] = { "always", "auto", "mini", "never" }, ["gui-position"] = { "none", "typ", "type", "name", "fix" } } do
		local setting = default_value(cfg_player_get(pid,option),"")
		tbl = flow.add { type = "flow", direction = "vertical" }
		tbl.style.vertical_spacing = 0
		e = tbl.add { type = "label", style = "label", caption = { "mod-setting-name.sbSortedLogistics-" .. option }, tooltip = { "mod-setting-description.sbSortedLogistics-" .. option } }
		for _, v in pairs(values) do
			e = tbl.add { type = "radiobutton", state = v == setting, caption = { "string-mod-setting.sbSortedLogistics-" .. option .. "-" .. v }, tooltip = { "string-mod-setting-description.sbSortedLogistics-" .. option .. "-" .. v } }
			e.tags = { ["on_click"] = { "set_config", "plr", "el", option, v } }
		end
	end
	e = strip.add { type = "line", direction = "horizontal" }
	e = strip.add { type = "button", caption = { "sbSortedLogistics-name.btn-set-default-pos" }, tooltip = { "sbSortedLogistics-desc.btn-set-default-pos" } }
	e.tags = { ["on_click"] = { "set_pos_default", "el" } }
	e.style.horizontally_stretchable = true
	e.style.horizontally_squashable = false
	e.style.minimal_width = 0
	e.style.minimal_height = 0
	e = strip.add { type = "button", caption = { "sbSortedLogistics-name.btn-reset-gui" }, tooltip = { "sbSortedLogistics-desc.btn-reset-gui" } }
	e.tags = { ["on_click"] = { "del_pos_data", "plr" } }
	e.style.horizontally_stretchable = true
	e.style.horizontally_squashable = false
	e.style.minimal_width = 0
	e.style.minimal_height = 0
	e = strip.add { type = "line", direction = "horizontal" }
	strip.visible = db.open_config

	return gui
end

---@param plr LuaPlayer
---@return nil
local function gui_destroy(plr)
	return gui_destroy_deep(plr.gui)
end

---@param gui LuaGuiElement
---@param elements table[]
---@return nil
local function gui_add_elements(gui, elements)
	for _, v in pairs(elements) do
		local style = v.style_overwrite
		local children = v.children
		if style then v.style_overwrite = nil end
		if children then v.children = nil end
		local el = gui.add(v)
		if style then
			local st = el.style
			for k, o in pairs(style) do
				st[k] = o
			end
		end
		if children then
			gui_add_elements(el, children)
		end
	end
end

---@param plr LuaPlayer
---@param elements table
---@param ntt LuaEntity|LuaItem|LuaInventory
---@param typ string
local function gui_show(plr,elements,ntt,typ)
	gui_destroy_deep(plr.gui)
	if #elements < 1 then
		return
	end
	local gui = gui_get_window(plr,ntt,typ)
	gui_add_elements(gui,elements)
	gui_enable_drag(gui)

	local pin = gui_locate_child(gui,"btn_pin_top")
	if pin and pin.toggled then
		get_db("foreground_gui")[gui] = pin
		start_foreground_gui()
	end
end

---@param el LuaGuiElement
---@param ntt LuaEntity|LuaItem|LuaInventory
---@param typ string
local function gui_moved(el,ntt,typ)
	local which = cfg_player_get(el.player_index,"gui-position")
	if which == "fix" then return end
	local db = get_db_player(el.player_index)
	local pos = hlp_pos_uniform(el.location)
	if which == "none" then
		db.pos_default = pos
		return
	end
	local ptype,pname = get_prototype_type(ntt)
	local dat = db.pos_data
	dat.typ[typ] = pos
	dat.type[ptype] = pos
	dat.name[pname] = pos
end

---@param plr LuaPlayer
---@param ntt LuaEntity
---@param typ string
---@param when string
---@return nil
local function gui_show_entity(plr, ntt, typ, when)
	gui_db_player_set(plr.index,ntt,typ)
	local elements = {}

	local more = false
	if typ then
		for k, v in pairs(cfg_inventories[ntt.type] or arr_empty) do
			if get_first_valid_inventory(ntt, v) and (when == "always" or not cfg_player_get(plr.index,arn_concat{"sort-",typ,k})) then
				gui_add_spacer(elements, "vertical", 5)
				gui_add_button(elements, typ .. k, { "sort_inv", "ntt", k })
				more = true
				if when ~= "mini" then
					gui_add_setting(plr, elements, typ .. k)
				end
			end
		end
		if more and when ~= "mini" then
			gui_add_spacer(elements, "vertical", 5)
			gui_add_button(elements, "allinv", { "sort_inv", "ntt" })
			local e = tbl_get_tbl(gui_add_flow(elements, "horizontal"), "children")
			gui_add_min_button(e, "inv-all", { "set_all_inv", "plr", true })
			gui_add_spacer(e, "horizontal")
			gui_add_min_button(e, "inv-none", { "set_all_inv", "plr", false })
		end
	end
	if has_sortable_logistic_sections(ntt) and (when == "always" or not cfg_player_get(plr.index,"sort-"..(typ or "other"))) then
		gui_add_spacer(elements, "vertical", 15)
		gui_add_button(elements, typ or "other", { "sort_logi", "ntt" })
		if when ~= "mini" then
			gui_add_setting(plr, elements, typ or "other")
		end
		local e = tbl_get_tbl(gui_add_flow(elements, "horizontal"), "children")
		gui_add_min_button(e, "logi-all", { "set_all_logi", "plr", true })
		gui_add_spacer(e, "horizontal")
		gui_add_min_button(e, "logi-none", { "set_all_logi", "plr", false })
	end
	if cfg_has_infinity_filter[ntt.type] and (when == "always" or not cfg_player_get(plr.index,"sort-infinity")) then
		gui_add_spacer(elements, "vertical", 15)
		gui_add_button(elements, "infinity", { "sort_infinity", "ntt" })
		if when ~= "mini" then
			gui_add_setting(plr, elements, "infinity")
		end
	end

	return gui_show(plr,elements,ntt,typ)
end

local function gui_show_planner(plr, itm, typ, when)
	gui_db_player_set(plr.index,itm,typ)
	local elements = {}

	if when == "always" or not cfg_player_get(plr,"sort-"..typ) or (typ == "book" and not cfg_player_get(plr,"sort-books")) then
		for _, t in pairs { typ, (typ == "book" and "books" or nil) } do
			gui_add_button(elements, t, { "sort_item", "plr", "ntt" })
		end
		if when ~= "mini" then
			for _, t in pairs(typ == "book" and { "decon", "upgrade", typ, "books" } or { typ }) do
				gui_add_setting(plr, elements, t)
			end
		end
	end

	return gui_show(plr,elements,itm,typ)
end

---@type table<string,function>
local gui_click = {
	["sort_inv"] = function(ntt, num)
		return handle_entity_inventory(ntt, num)
	end,
	["sort_logi"] = function(ntt)
		return handle_entity_logistic_sections(ntt)
	end,
	["sort_infinity"] = function(ntt)
		return handle_entity_infinity_filters(ntt)
	end,
	["sort_item"] = function(plr,itm)
		return actn_sort_object(plr.index,itm)
	end,
	---@param el LuaGuiElement
	["btn_pin_top"] = function(el)
		local val = el.toggled
		get_db(el.player_index).pin_top = val
		if not val then return end
		local gui = gui_locate_parent(el,"sbSortedLogistics_gui")
		if not gui then return end
		get_db("foreground_gui")[gui] = el
		start_foreground_gui()
	end,
	---@param el LuaGuiElement
	["btn_config"] = function(el)
		local val = el.toggled
		get_db(el.player_index).open_config = val
		local gui = gui_locate_parent(el,"sbSortedLogistics_gui")
		if not gui then return end
		gui["config"].visible = val
	end,
	---@param el LuaGuiElement
	["set_pos_default"] = function(el)
		local gui = gui_locate_parent(el,"sbSortedLogistics_gui")
		if not gui then return end
		get_db(el.player_index).pos_default = hlp_pos_uniform(gui.location)
	end,
	---@param plr LuaPlayer
	["del_pos_data"] = function(plr)
		local db = get_db(plr.index)
		db.pos_default = {0,0}
		db.pos_data = {["typ"]={},["type"]={},["name"]={}}
		return gui_destroy_deep(plr.gui)
	end,
	---@param plr LuaPlayer
	---@param el LuaGuiElement
	---@param option string
	---@param value any
	["set_config"] = function(plr, el, option, value)
		if value == nil then value = el.state end
		cfg_player_set(plr.index,option,value)
	end,
	["set_all_inv"] = function(plr, value)
		for _, opt in pairs(cfg_inv_options) do
			cfg_player_set(plr.index,"sort-"..opt,value)
		end
	end,
	["set_all_logi"] = function(plr, value)
		for _, opt in pairs(cfg_logi_options) do
			cfg_player_set(plr.index,"sort-"..opt,value)
		end
	end,
}





-- handle gui open and auto sort
local function evt_handle_entity(plr, open, ntt, typ)
	local pid = plr.index
	if cfg_player_get(pid,"enabled") then
		actn_sort_object(pid,ntt)
	end
	if not open then return end
	local when = default_value(cfg_player_get(pid,"gui-show"),"always")
	if when == "never" then return end
	return gui_show_entity(plr, ntt, typ, when)
end
---@param open boolean?
---@param plr LuaPlayer
---@param itm LuaItemStack|LuaRecord|LuaInventory
---@param typ string
---@return nil
local function evt_handle_planner(open,plr,itm,typ)
	local pid = plr.index
	if cfg_player_get(pid,"enabled") then
		actn_sort_object(pid,itm)
	end
	if not open then return end
	local when = default_value(cfg_player_get(pid,"gui-show"),"always")
	if when == "never" then return end
	return gui_show_planner(plr, itm, typ, when)
end






-- events --

---@param ev event.on_runtime_mod_setting_changed
script.on_event(defines.events.on_runtime_mod_setting_changed, safecall_wrapper(function(ev)
	if ev.setting_type ~= "runtime-per-user" then return end
	local pid = ev.player_index
	---@cast pid -nil
	local setting = ev.setting
	local value = cfg_player_get_raw(pid,setting)
	if setting == "sbSortedLogistics-enabled" then
		return evt_sync_autosort(pid,value and true or false)
	end
	local gui = get_player(pid).gui.screen["sbSortedLogistics_gui"]
	if not gui then return end
	---@type string
	local option = str_substr(setting, 19)
	return gui_recurse_children(gui, function(el)
		---@type table
		local tags = el.tags
		if not tags then return end
		tags = tags.on_click
		if not tags or tags[1] ~= "set_config" or tags[4] ~= option then return end
		if el.type == "radiobutton" then
			el.state = tags[5] == value
		elseif el.type == "checkbox" then
			---@cast value boolean
			el.state = value
		end
	end)
end))

---@param ev event.on_gui_location_changed
script.on_event(defines.events.on_gui_location_changed,safecall_wrapper(function(ev)
	local el = ev.element
	if el.name~="sbSortedLogistics_gui" then return end
	local ntt,typ = gui_db_player_get(ev.player_index)
	if not ntt or not typ then return end
	return gui_moved(el,ntt,typ)
end))

---@param ev event.on_gui_click
script.on_event(defines.events.on_gui_click, safecall_wrapper(function(ev)
	---@type LuaGuiElement
	local el = ev.element
	if not gui_locate_parent(el, "sbSortedLogistics_gui") then return end
	---@type any
	local actn = el.tags
	if not actn then return end
	actn = actn["on_click"]
	if not actn then return end
	local fn = gui_click[arr_remove(actn, 1)]
	if not fn then return end
	local ntt = gui_db_player_get(ev.player_index)
	local tl = {
		["plr"] = get_player(ev.player_index) or false,
		["el"] = el or false,
		["ntt"] = ntt and ntt.valid and ntt or false,
	}
	for k, v in pairs(actn) do
		v = tl[v]
		if v == false then return end
		if v then actn[k] = v end
	end
	return fn(arn_unpack(actn))
end))

do
-- branch out gui open close event --
local cfg_defines_controllers_character = defines.controllers.character
---@type { [defines.gui_type]: fun(ev:event.on_gui_opened,plr:LuaPlayer,open:boolean) }
local evt_gui_open_close_func = {
	[defines.gui_type.controller] = function(_, plr, open)
		local chr = plr.character
		if chr and plr.controller_type == cfg_defines_controllers_character then
			return evt_handle_entity(plr, open, chr, "self")
		end
	end,
	[defines.gui_type.entity] = function(ev, plr, open)
		local ntt = ev.entity
		if not ntt then return end
		return evt_handle_entity(plr, open, ntt, cfg_type_map[ntt.type])
	end,
	[defines.gui_type.item] = function(ev, plr, open)
		local itm = ev.item
		if not itm then return end
		local typ = cfg_type_map[itm.type]
		if typ then
			return evt_handle_planner(open, plr, itm, typ)
		end
	end,
	[defines.gui_type.script_inventory] = function(ev, plr, open)
		return evt_handle_planner(open, plr, ev.inventory, "scriptinv")
	end,
}
---@param ev event.on_gui_opened
script.on_event(defines.events.on_gui_opened, safecall_wrapper(function(ev)
	---@type LuaPlayer
	local plr = get_player(ev.player_index)
	gui_destroy(plr)
	gui_db_player_set(plr.index)
	local fn = evt_gui_open_close_func[ev.gui_type]
	if fn then return fn(ev, plr, true) end
end))
---@type { [string]: fun(plr:LuaPlayer,obj:any) }
local handle_opened = {
	---@param obj LuaEntity
	["LuaEntity"] = function(plr, obj)
		return evt_handle_entity(plr, true, obj, cfg_type_map[obj.type])
	end,
	---@param obj LuaItemStack
	["LuaItemStack"] = function(plr, obj)
		local typ = cfg_type_map[obj.type]
		if typ then
			return evt_handle_planner(true, plr, obj, typ)
		end
	end,
	---@param obj LuaInventory
	["LuaInventory"] = function(plr, obj)
		return evt_handle_planner(true, plr, obj, "scriptinv")
	end,
}
---@param ev event.on_gui_closed
script.on_event(defines.events.on_gui_closed, safecall_wrapper(function(ev)
	---@type LuaPlayer
	local plr = get_player(ev.player_index)
	gui_destroy(plr)
	gui_db_player_set(plr.index)
	---@type any
	local fn = evt_gui_open_close_func[ev.gui_type]
	if fn then
		fn(ev,plr,false)
	end
	if plr.opened_self then
		fn = plr.character
		if fn and plr.controller_type == cfg_defines_controllers_character then
			return evt_handle_entity(plr,true,fn,"self")
		end
		return
	end
	local obj = plr.opened
	if obj then
		fn = handle_opened[obj.object_name]
		if fn then
			return fn(plr,obj)
		end
	end
end))
end

---@param ev event.on_player_cursor_stack_changed
script.on_event(defines.events.on_player_cursor_stack_changed, safecall_wrapper(function(ev)
	---@type LuaPlayer
	local plr = get_player(ev.player_index)
	---@type string
	local typ
	---@type LuaRecord|LuaItemStack|nil
	local itm = plr.cursor_record
	if itm and itm.valid_for_write then
		typ = get_prototype_type(itm)
	else
		itm = plr.cursor_stack
		if itm and itm.valid_for_read then
			typ = itm.type
		end
	end
	typ = cfg_type_map[typ]
	if typ then
		---@cast itm -nil
		return evt_handle_planner(nil, plr, itm, typ)
	end
end))

---@param ev event.on_player_created
script.on_event(defines.events.on_player_created, safecall_wrapper(function(ev)
	local pid = ev.player_index
	return evt_sync_autosort(pid,cfg_player_get_raw(pid,"sbSortedLogistics-enabled") and true or false)
end))

---@param ev event.on_player_removed
script.on_event(defines.events.on_player_removed, safecall_wrapper(function(ev)
	get_db(ev.player_index,true)
end))

script.on_load(safecall_wrapper(function()
	if storage and storage["foreground_gui"] and tbl_size(storage["foreground_gui"]) >= 1 then
		start_foreground_gui()
	end
end))

---@param ev ConfigurationChangedData
script.on_configuration_changed(safecall_wrapper(function(ev)
	local ver = ev.mod_changes[mod_name]
	if not ver or not ver.old_version then return end
	local old_ver = hlp_ver_explode(ver.old_version)
	if cmp_arr(old_ver,{1,0,11}) then
		local pd = get_db("player_data")
		for pid,plr in pairs(game.players) do
			---@cast pid integer
			local db = get_db_player(pid)
			if pd[pid] then
				db.gui_ntt = pd[pid][1]
				db.gui_type = pd[pid][2]
			end
			---@type LuaGuiElement
			local gui = plr.gui.screen["sbSortedLogistics_gui"]
			if gui then
				local tags = gui.tags
				if tags then
					---@diagnostic disable-next-line:assign-type-mismatch
					db.pos_default = tags["pos_default"] or {0,0}
					---@diagnostic disable-next-line:assign-type-mismatch
					db.pos_data = {["typ"]=tags["typ"] or {},["type"]=tags["type"] or tags[mod_name] or {},["name"]=tags["name"] or {}}
				end
				local pin = gui_locate_child(gui,"btn_pin_top")
				if pin then
					db.pin_top = pin.toggled
				end
			end
			gui_destroy_deep(plr.gui)
		end
		get_db("player_data",true)
	end
end))

do
---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-sort-self",safecall_wrapper(function(ev)
	return evt_sort_self(ev.player_index,false)
end))
---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-sort-self-full",safecall_wrapper(function(ev)
	return evt_sort_self(ev.player_index,true)
end))
---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-sort-opened",safecall_wrapper(function(ev)
	return evt_sort_opened(ev.player_index,false)
end))
---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-sort-opened-full",safecall_wrapper(function(ev)
	return evt_sort_opened(ev.player_index,true)
end))
---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-sort-hand",safecall_wrapper(function(ev)
	return evt_sort_hand(ev.player_index,false)
end))
---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-sort-hand-full",safecall_wrapper(function(ev)
	return evt_sort_hand(ev.player_index,true)
end))
---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-sort-target",safecall_wrapper(function(ev)
	return evt_sort_target(ev.player_index,false)
end))
---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-sort-target-full",safecall_wrapper(function(ev)
	return evt_sort_target(ev.player_index,true)
end))
end

do
---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-toggle",safecall_wrapper(function(ev)
	return evt_toggle_autosort(ev.player_index)
end))

---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-enable",safecall_wrapper(function(ev)
	return evt_toggle_autosort(ev.player_index,true)
end))

---@param ev event.CustomInputEvent
script.on_event("sbSortedLogistics-disable",safecall_wrapper(function(ev)
	return evt_toggle_autosort(ev.player_index,false)
end))

---@type { [string]: fun(pid:integer) }
local on_lua_shortcut_func = {
	["sbSortedLogistics-toggle"] = evt_toggle_autosort,
}
---@param ev event.on_lua_shortcut
script.on_event(defines.events.on_lua_shortcut,safecall_wrapper(function(ev)
	local fn = on_lua_shortcut_func[ev.prototype_name]
	if fn then return fn(ev.player_index) end
end))
end

do
---@type { [string]: fun(pid:integer,txt:string) }
local command = {
	["help"] = function(pid)
		local print = pid and get_player(pid).print or game.print
		print({ "sbSortedLogistics-command.help-text" })
	end,
	["reset"] = function(pid)
		if not pid then return end
		local plr = get_player(pid)
		gui_destroy_deep(plr.gui)
		local db = get_db_player(pid)
		db.pos_default = {0,0}
		db.pos_data = {["typ"]={},["type"]={},["name"]={}}
		plr.print({ "sbSortedLogistics-command.reset-player" })
	end,
}
---@param ev CustomCommandData
commands.add_command("sortme", { "sbSortedLogistics-command.help-text" }, safecall_wrapper(function(ev)
	local c, p = re_match(ev.parameter or "", "^[\x00-\x20]*([^\x00-\x20]*)[\x00-\x20]*(.-)[\x00-\x20]*$")
	return (command[c] or command.help)(ev.player_index, p)
end))
end